/**
 * 直線判定クラス
 * 
 * @constructor
 */
function ThreeEyeGameLineJudge() {
	/**
	 * 揃った方向
	 * 
	 * @type {number}
	 */
	this.comp_direction = THREE_EYE_GAME_DEFINE.DIRECTION_NONE;
}

/**
 * マスを指定して、8方向を調べて揃っているか判定する。
 * 
 * @param {ThreeEyeGameBoardData}
 *            board 盤データ
 * @param {ThreeEyeGameMassData}
 *            mass_data マスデータ
 * @param {number}
 *            comp_num 何個揃えばいいかの個数
 * @returns {number} 揃った方向
 */
ThreeEyeGameLineJudge.prototype.judgeMassLineComp = function(board, mass_data,
		comp_num) {

	if (this.isDirectionComp(board, mass_data,
			THREE_EYE_GAME_DEFINE.DIRECTION_UP, comp_num)) {

	} else if (this.isDirectionComp(board, mass_data,
			THREE_EYE_GAME_DEFINE.DIRECTION_DOWN, comp_num)) {

	} else if (this.isDirectionComp(board, mass_data,
			THREE_EYE_GAME_DEFINE.DIRECTION_LEFT, comp_num)) {

	} else if (this.isDirectionComp(board, mass_data,
			THREE_EYE_GAME_DEFINE.DIRECTION_RIGHT, comp_num)) {

	} else if (this.isDirectionComp(board, mass_data,
			THREE_EYE_GAME_DEFINE.DIRECTION_LEFT_UP, comp_num)) {

	} else if (this.isDirectionComp(board, mass_data,
			THREE_EYE_GAME_DEFINE.DIRECTION_RIGHT_UP, comp_num)) {

	} else if (this.isDirectionComp(board, mass_data,
			THREE_EYE_GAME_DEFINE.DIRECTION_LEFT_DOWN, comp_num)) {

	} else if (this.isDirectionComp(board, mass_data,
			THREE_EYE_GAME_DEFINE.DIRECTION_RIGHT_DOWN, comp_num)) {

	} else {

	}
	return this.getCompDirection();
};

/**
 * 方向判定
 * 
 * @param {ThreeEyeGameBoardData}
 *            board 盤データ
 * @param {ThreeEyeGameMassData}
 *            start_mass マスデータ
 * @param {number}
 *            direction 方向
 * @param {number}
 *            comp_num 何個揃えばいいかの個数
 * @returns {boolean} 揃っているかどうか
 */
ThreeEyeGameLineJudge.prototype.isDirectionComp = function(board, start_mass,
		direction, comp_num) {

	var count = this.getDirectionCompNum(board, start_mass, direction);

	// 同じ状態が渡された個数以上だったら方向を返す。
	if (count >= comp_num) {
		this.comp_direction = direction;
		return true;
	} else {
		return false;
	}
};

/**
 * 揃っているマスの数を取得
 * 
 * @param {ThreeEyeGameBoardData}
 *            board 盤データ
 * @param {ThreeEyeGameMassData}
 *            start_mass マスデータ
 * @param {number}
 *            direction 方向
 * @returns {number} 揃っている数取得
 */
ThreeEyeGameLineJudge.prototype.getDirectionCompNum = function(board,
		start_mass, direction) {

	// マス状態を取得
	var start_mass_status = start_mass.getMassStatus();

	// プレイヤーが置いてなければ終了
	if (start_mass_status <= THREE_EYE_GAME_DEFINE.MASS_STATUS_SPASE) {
		return 0;
	}

	// すでにプレイヤーが置かれているので初期値は1
	var count = 1;

	// 渡された方向にマスを調べる
	for ( var mass = board.getDirectionMass(start_mass, direction); mass != null; mass = board
			.getDirectionMass(mass, direction)) {
		// マスが渡されたマス状態と同じか判定
		var judge_status = mass.getMassStatus();
		if (judge_status == start_mass_status) {
			count++;
		} else {
			break;
		}
	}

	return count;
};

/**
 * 揃っているマスの数を取得
 * スペースマスは無視して進む
 * 
 * @param {ThreeEyeGameBoardData}
 *            board 盤データ
 * @param {ThreeEyeGameMassData}
 *            start_mass マスデータ
 * @param {number}
 *            direction 方向
 * @returns {number} 揃っている数取得
 */
ThreeEyeGameLineJudge.prototype.getDirectionCompNumSpaceIgnore = function(board,
		start_mass, direction) {

	// マス状態を取得
	var start_mass_status = start_mass.getMassStatus();

	// プレイヤーが置いてなければ終了
	if (start_mass_status <= THREE_EYE_GAME_DEFINE.MASS_STATUS_SPASE) {
		return 0;
	}

	// すでにプレイヤーが置かれているので初期値は1
	var count = 1;

	// 渡された方向にマスを調べる
	for ( var mass = board.getDirectionMass(start_mass, direction); mass != null; mass = board
			.getDirectionMass(mass, direction)) {
		// マスが渡されたマス状態と同じか判定
		var judge_status = mass.getMassStatus();
		if (judge_status == start_mass_status) {
			count++;
		} 
		else if(judge_status == THREE_EYE_GAME_DEFINE.MASS_STATUS_SPASE){
			// 次のマスへ進む
		}
		else {
			break;
		}
	}

	return count;
};

/**
 * 8方向で一番多い数を取得
 * 
 * @param {ThreeEyeGameBoardData}
 *            board 盤データ
 * @param {ThreeEyeGameMassData}
 *            start_mass マスデータ
 * @returns {number} 8方向で一番多い数
 */
ThreeEyeGameLineJudge.prototype.getMaxCompNum = function(board, start_mass) {

	var max_count = 0;

	// 方向の数だけループ
	// ※方向を列挙型で宣言したい
	for ( var direction = 1; direction < 9; direction++) {
		var count = this.getDirectionCompNum(board, start_mass, direction);
		if (max_count < count) {
			max_count = count;
		}
	}

	return max_count;
};

/**
 * 揃った方向取得
 * 
 * @returns {number} 揃った方向
 */
ThreeEyeGameLineJudge.prototype.getCompDirection = function() {
	return this.comp_direction;
};

/**
 * 水平方向の個数を取得
 * @param {ThreeEyeGameBoardData} board 盤データ
 * @param {ThreeEyeGameMassData}
 *            start_mass マスデータ
 * @returns {number} 水平方向の個数
 */
ThreeEyeGameLineJudge.prototype.getCompNumHorizonSpaceIgnore = function(board, start_mass){
	
	// マス状態を取得
	var start_mass_status = start_mass.getMassStatus();

	// プレイヤーが置いてなければ終了
	if (start_mass_status <= THREE_EYE_GAME_DEFINE.MASS_STATUS_SPASE) {
		return 0;
	}	
	
	var count = this.getDirectionCompNumSpaceIgnore(board, start_mass, THREE_EYE_GAME_DEFINE.DIRECTION_LEFT);
	count += this.getDirectionCompNumSpaceIgnore(board, start_mass, THREE_EYE_GAME_DEFINE.DIRECTION_RIGHT);
	
	count--;
	
	return count;
};

/**
 * 垂直方向の個数を取得
 * @param {ThreeEyeGameBoardData} board 盤データ
 * @param {ThreeEyeGameMassData}
 *            start_mass マスデータ
 * @returns {number} 垂直方向の個数
 */
ThreeEyeGameLineJudge.prototype.getCompNumVerticalSpaceIgnore = function(board, start_mass){
	
	// マス状態を取得
	var start_mass_status = start_mass.getMassStatus();

	// プレイヤーが置いてなければ終了
	if (start_mass_status <= THREE_EYE_GAME_DEFINE.MASS_STATUS_SPASE) {
		return 0;
	}	
	
	var count = this.getDirectionCompNumSpaceIgnore(board, start_mass, THREE_EYE_GAME_DEFINE.DIRECTION_UP);
	count += this.getDirectionCompNumSpaceIgnore(board, start_mass, THREE_EYE_GAME_DEFINE.DIRECTION_DOWN);
	
	count--;

	return count;
};

/**
 * 左上から右下にかけての個数を取得
 * @param {ThreeEyeGameBoardData} board 盤データ
 * @param {ThreeEyeGameMassData}
 *            start_mass マスデータ
 * @returns {number} 左上から右下にかけての個数
 */
ThreeEyeGameLineJudge.prototype.getCompNumLeftUpSpaceIgnore = function(board, start_mass){
	
	// マス状態を取得
	var start_mass_status = start_mass.getMassStatus();

	// プレイヤーが置いてなければ終了
	if (start_mass_status <= THREE_EYE_GAME_DEFINE.MASS_STATUS_SPASE) {
		return 0;
	}	
	
	var count = this.getDirectionCompNumSpaceIgnore(board, start_mass, THREE_EYE_GAME_DEFINE.DIRECTION_LEFT_UP);
	count += this.getDirectionCompNumSpaceIgnore(board, start_mass, THREE_EYE_GAME_DEFINE.DIRECTION_RIGHT_DOWN);
	
	count--;

	return count;
};

/**
 * 右上から左下にかけての個数を取得
 * @param {ThreeEyeGameBoardData} board 盤データ
 * @param {ThreeEyeGameMassData}
 *            start_mass マスデータ
 * @returns {number} 右上から左下にかけての個数
 */
ThreeEyeGameLineJudge.prototype.getCompNumRightUpSpaceIgnore = function(board, start_mass){
	
	// マス状態を取得
	var start_mass_status = start_mass.getMassStatus();

	// プレイヤーが置いてなければ終了
	if (start_mass_status <= THREE_EYE_GAME_DEFINE.MASS_STATUS_SPASE) {
		return 0;
	}	
	
	var count = this.getDirectionCompNumSpaceIgnore(board, start_mass, THREE_EYE_GAME_DEFINE.DIRECTION_RIGHT_UP);
	count += this.getDirectionCompNumSpaceIgnore(board, start_mass, THREE_EYE_GAME_DEFINE.DIRECTION_LEFT_DOWN);
	
	count--;
	
	return count;
};

/**
 * 水平方向の個数を取得
 * @param {ThreeEyeGameBoardData} board 盤データ
 * @param {ThreeEyeGameMassData}
 *            start_mass マスデータ
 * @returns {number} 水平方向の個数
 */
ThreeEyeGameLineJudge.prototype.getCompNumHorizon = function(board, start_mass){
	
	// マス状態を取得
	var start_mass_status = start_mass.getMassStatus();

	// プレイヤーが置いてなければ終了
	if (start_mass_status <= THREE_EYE_GAME_DEFINE.MASS_STATUS_SPASE) {
		return 0;
	}	
	
	var count = this.getDirectionCompNum(board, start_mass, THREE_EYE_GAME_DEFINE.DIRECTION_LEFT);
	count += this.getDirectionCompNum(board, start_mass, THREE_EYE_GAME_DEFINE.DIRECTION_RIGHT);
	
	count--;
	
	return count;
};

/**
 * 垂直方向の個数を取得
 * @param {ThreeEyeGameBoardData} board 盤データ
 * @param {ThreeEyeGameMassData}
 *            start_mass マスデータ
 * @returns {number} 垂直方向の個数
 */
ThreeEyeGameLineJudge.prototype.getCompNumVertical = function(board, start_mass){
	
	// マス状態を取得
	var start_mass_status = start_mass.getMassStatus();

	// プレイヤーが置いてなければ終了
	if (start_mass_status <= THREE_EYE_GAME_DEFINE.MASS_STATUS_SPASE) {
		return 0;
	}	
	
	var count = this.getDirectionCompNum(board, start_mass, THREE_EYE_GAME_DEFINE.DIRECTION_UP);
	count += this.getDirectionCompNum(board, start_mass, THREE_EYE_GAME_DEFINE.DIRECTION_DOWN);
	
	count--;

	return count;
};

/**
 * 左上から右下にかけての個数を取得
 * @param {ThreeEyeGameBoardData} board 盤データ
 * @param {ThreeEyeGameMassData}
 *            start_mass マスデータ
 * @returns {number} 左上から右下にかけての個数
 */
ThreeEyeGameLineJudge.prototype.getCompNumLeftUp = function(board, start_mass){
	
	// マス状態を取得
	var start_mass_status = start_mass.getMassStatus();

	// プレイヤーが置いてなければ終了
	if (start_mass_status <= THREE_EYE_GAME_DEFINE.MASS_STATUS_SPASE) {
		return 0;
	}	
	
	var count = this.getDirectionCompNum(board, start_mass, THREE_EYE_GAME_DEFINE.DIRECTION_LEFT_UP);
	count += this.getDirectionCompNum(board, start_mass, THREE_EYE_GAME_DEFINE.DIRECTION_RIGHT_DOWN);
	
	count--;

	return count;
};

/**
 * 右上から左下にかけての個数を取得
 * @param {ThreeEyeGameBoardData} board 盤データ
 * @param {ThreeEyeGameMassData}
 *            start_mass マスデータ
 * @returns {number} 右上から左下にかけての個数
 */
ThreeEyeGameLineJudge.prototype.getCompNumRightUp = function(board, start_mass){
	
	// マス状態を取得
	var start_mass_status = start_mass.getMassStatus();

	// プレイヤーが置いてなければ終了
	if (start_mass_status <= THREE_EYE_GAME_DEFINE.MASS_STATUS_SPASE) {
		return 0;
	}	
	
	var count = this.getDirectionCompNum(board, start_mass, THREE_EYE_GAME_DEFINE.DIRECTION_RIGHT_UP);
	count += this.getDirectionCompNum(board, start_mass, THREE_EYE_GAME_DEFINE.DIRECTION_LEFT_DOWN);
	
	count--;
	
	return count;
};

/**
 * 水平・垂直・ななめ方向の中で、一番多い方向で揃っている個数と、
 * 一番多い個数が何方向あるか取得する。
 * @param {ThreeEyeGameBoardData} board 盤データ
 * @param {ThreeEyeGameMassData}
 *            start_mass マスデータ
 * @returns {JugeLineResult} 直線調査結果
 */
ThreeEyeGameLineJudge.prototype.getLineNumResult = function(board, start_mass){

	var max_count = 0;
	var count_h = this.getCompNumHorizon(board, start_mass);
	if(count_h > max_count){
		max_count = count_h;
	}
	var count_v = this.getCompNumVertical(board, start_mass);
	if(count_v > max_count){
		max_count = count_v;
	}
	var count_l = this.getCompNumLeftUp(board, start_mass);
	if(count_l > max_count){
		max_count = count_l;
	}
	var count_r = this.getCompNumRightUp(board, start_mass);
	if(count_r > max_count){
		max_count = count_r;
	}
	
	var line_count = 0;
	if(max_count == count_h){
		line_count++;
	}
	if(max_count == count_v){
		line_count++;
	}
	if(max_count == count_l){
		line_count++;
	}
	if(max_count == count_r){
		line_count++;
	}
	
	return new JugeLineResult(max_count, line_count, start_mass.getMassStatus());
};

/**
 * 直線調査結果
 * @param {number} max_count 最大値
 * @param {number} line_num 直線数
 * @param {number} plyaer_id プレイヤー識別子
 * @constructor
 */
function JugeLineResult(max_count, line_num, plyaer_id){
	/**
	 * 最大数
	 * @type {number}
	 */
	this.max_count = max_count;
	/**
	 * 直線数
	 * @type {number}
	 */
	this.line_num = line_num;
	/**
	 * プレイヤーID
	 * @type {number}
	 */
	this.player_id = plyaer_id;
}
