/**
 * 三目並べゲーム管理クラス
 * 
 * @constructor
 */
function ThreeEyeGameManage() {
	/**
	 * シーン
	 * 
	 * @type {number}
	 */
	this.scene = THREE_EYE_GAME_DEFINE.SCENE_PLAY;

	/**
	 * プレイヤー配列
	 * 
	 * @type {Array.<ThreeEyeGamePlayer>}
	 */
	this.playerArray = [];
	
	/**
	 * 解析
	 * @type {ThreeEyeGameAnalyzer}
	 */
	this.analyze = new ThreeEyeGameAnalyzer();
	
	/**
	 * 勝者
	 * @type {ThreeEyeGamePlayer}
	 */
	this.winner = null;

	// プレイヤーを初期化
	this.initPlayer();
}

/**
 * プレイヤー初期化
 */
ThreeEyeGameManage.prototype.initPlayer = function() {

	this.playerArray = [];

	for ( var ii = 0; ii < g_control.getPlayerNum(); ii++) {
		var operation = g_control.getPlayerOperation(ii);
		this.playerArray[ii] = new ThreeEyeGamePlayer(ii + 1, operation);
	}
	// 先頭のプレイヤーをプレイ中にする
	this.playerArray[0].setCurrent(true);
};

/**
 * クリックハンドラー シーンによって動きを変える。
 * 
 * @param {Event}
 *            e マウスイベント
 * @param {ThreeEyeGameData}
 *            data 三目並べゲームデータ
 * @param {ThreeEyeGameDrawer}
 *            drawer 三目並べゲーム描画
 */
ThreeEyeGameManage.prototype.clickHandler = function(e, data, drawer) {

	if (this.scene == THREE_EYE_GAME_DEFINE.SCENE_PLAY) {
		// 試合中
		if(this.getCurrentPlayer().getOperation() != THREE_EYE_GAME_DEFINE.PLAY_COM){
			this.actionPlay(e, data);
		}else{
			this.actionComPlay(data);
		}
	} else if (this.scene == THREE_EYE_GAME_DEFINE.SCENE_FINISH) {
		// 終了状態
		this.actionFinish(data);
	}

};

/**
 * 試合中動作
 * 
 * @param {Event}
 *            e マウスイベント
 * @param {ThreeEyeGameData}
 *            data 三目並べゲームデータ
 */
ThreeEyeGameManage.prototype.actionPlay = function(e, data) {
	// クリックマス取得
	var mass_data = this.getClickMass(e, data);
	if (mass_data == null) {
		return;
	}

	// 置けないマスなら終了
	if (mass_data.getMassStatus() != THREE_EYE_GAME_DEFINE.MASS_STATUS_SPASE) {
		return;
	}

	// プレイヤー取得
	var player = this.getCurrentPlayer();
	// プレイヤー配置
	mass_data.setMassStatus(player.getId());

	// 試合終了か判定
	this.judgeFinish(data, g_control.getCompNum());
	// 終了なら終了
	if (this.scene == THREE_EYE_GAME_DEFINE.SCENE_FINISH) {
		return;
	}

	// 次のプレイヤーを設定
	this.setNextPlayer();
};

/**
 * コンピュータの番
 * 
 * @param {ThreeEyeGameData}
 *            data 三目並べゲームデータ
 */
ThreeEyeGameManage.prototype.actionComPlay = function(data) {

	// プレイヤー取得
	var player = this.getCurrentPlayer();

	// COM以外なら終了
	if(player.getOperation() != THREE_EYE_GAME_DEFINE.PLAY_COM)
	{
		return;
	}

	var mass_data = this.analyze.getPutMass(this, data);
	// プレイヤー配置
	mass_data.setMassStatus(player.getId());

	// 試合終了か判定
	this.judgeFinish(data, g_control.getCompNum());
	// 終了なら終了
	if (this.scene == THREE_EYE_GAME_DEFINE.SCENE_FINISH) {
		return;
	}

	// 次のプレイヤー配置
	this.setNextPlayer();

	player = this.getCurrentPlayer();

};

/**
 * 終了時動作
 * 
 * @param {ThreeEyeGameData}
 *            data 三目並べゲームデータ
 */
ThreeEyeGameManage.prototype.actionFinish = function(data) {
	// 盤データ初期化
	data.initBoardData();
	// 試合中シーンへ移行
	this.scene = THREE_EYE_GAME_DEFINE.SCENE_PLAY;
	// プレイヤー初期化
	this.initPlayer();
};

/**
 * プレイ中プレイヤー取得
 * 
 * @returns {ThreeEyeGamePlayer} プレイヤー
 */
ThreeEyeGameManage.prototype.getCurrentPlayer = function() {
	for ( var ii = 0; ii < this.playerArray.length; ii++) {
		if (this.playerArray[ii].isCurrent()) {
			return this.playerArray[ii];
		}
	}
	return null;
};

/**
 * クリックマス取得
 * 
 * @param {Event}
 *            e マウスイベント
 * @param {ThreeEyeGameData}
 *            data 三目並べゲームデータ
 * @returns {ThreeEyeGameMassData} マスデータ
 */
ThreeEyeGameManage.prototype.getClickMass = function(e, data) {

	// 盤データ取得
	var board_data = data.getBoardData();

	// 盤のサイズ取得
	var height_index = board_data.getHeightLength();
	var width_index = board_data.getWidhtLength();

	// 範囲内判定にコンテキストのisPointInPath()を使用する
	var mass_drawer = new ThreeEyeGameMassDrawer();

	for ( var ii = 0; ii < height_index; ii++) {
		for ( var jj = 0; jj < width_index; jj++) {
			// マス取得
			var mass = board_data.getMassData(jj, ii);
			// クリックされたか判定する
			var isClick = mass_drawer.isClickMass(e, mass);
			if (isClick) {
				return mass;
			}
		}
	}

	return null;

};

/**
 * 次プレイヤー設定
 */
ThreeEyeGameManage.prototype.setNextPlayer = function() {
	for ( var ii = 0; ii < this.playerArray.length; ii++) {
		if (this.playerArray[ii].isCurrent()) {
			// 現在プレイヤーをfalse
			this.playerArray[ii].setCurrent(false);

			var next_index = ii + 1;
			if (next_index >= this.playerArray.length) {
				next_index = 0;
			}
			// 次のプレイヤーをプレイ中に
			this.playerArray[next_index].setCurrent(true);

			return;
		}
	}
};

/**
 * 試合終了か判定する。
 * 
 * @param {ThreeEyeGameData}
 *            data 三目並べゲームデータ
 * @param {number}
 *            comp_num 何個揃えばいいかの個数
 */
ThreeEyeGameManage.prototype.judgeFinish = function(data, comp_num) {

	// 盤データ取得
	var board_data = data.getBoardData();

	// 盤サイズ取得
	var h_len = board_data.getHeightLength();
	var w_len = board_data.getWidhtLength();

	// 何も置かれていないマスのカウンタ
	var space_cnt = 0;

	for ( var ii = 0; ii < h_len; ii++) {
		for ( var jj = 0; jj < w_len; jj++) {
			// マスデータ取得
			var mass_data = board_data.getMassData(jj, ii);

			// スペースかどうか
			if (mass_data.getMassStatus() == THREE_EYE_GAME_DEFINE.MASS_STATUS_SPASE) {
				// スペースの数をインクリメント
				space_cnt++;
			}

			// 直線で揃っているか判定する
			var line_judge = new ThreeEyeGameLineJudge();
			var comp_direction = line_judge.judgeMassLineComp(board_data,
					mass_data, comp_num);
			if (comp_direction != THREE_EYE_GAME_DEFINE.DIRECTION_NONE) {
				// 揃ったマスの背景色を変更する。
				var line_setting = new ThreeEyeGameLineSetting();
				line_setting.setBackColor(board_data, mass_data,
						comp_direction, comp_num,
						THREE_EYE_GAME_DEFINE.COLOR_RED);
				// 終了シーンへ移行
				this.scene = THREE_EYE_GAME_DEFINE.SCENE_FINISH;
				// 勝者設定
				this.winner = this.getCurrentPlayer();
				
				return;
			}
		}
	}

	// スペースの数が0個なら終了
	if (space_cnt == 0) {
		this.scene = THREE_EYE_GAME_DEFINE.SCENE_FINISH;
		// 勝者なし
		this.winner = null;
	}

};

/**
 * プレイヤー取得
 * @param {number} index インデックス
 * @returns {ThreeEyeGamePlayer} プレイヤー
 */
ThreeEyeGameManage.prototype.getPlayer = function(index){
	return this.playerArray[index];
};

/**
 * タイマーを起動するかどうか
 * @returns {boolean} タイマーを起動するかどうか
 */
ThreeEyeGameManage.prototype.isTimerStart = function(){
	if(this.scene == THREE_EYE_GAME_DEFINE.SCENE_FINISH){
		return false;
	}
	
	if(this.getCurrentPlayer().getOperation() == THREE_EYE_GAME_DEFINE.PLAY_SELF){
		return false;
	}
	
	return true;
};

/**
 * シーン取得
 * @returns {number} シーン
 */
ThreeEyeGameManage.prototype.getScene = function(){
	return this.scene;
};

/**
 * 勝者取得
 * @returns {ThreeEyeGamePlayer} 勝者
 */
ThreeEyeGameManage.prototype.getWinner = function(){
	return this.winner;
};
