/**
 * 三目並べゲームマス描画
 * 
 * @constructor
 */
function ThreeEyeGameMassDrawer() {
	/**
	 * コンテキスト
	 * 
	 * @type {CanvasRenderingContext2D}
	 */
	this.ctx = document.getElementById(THREE_EYE_GAME_DEFINE.CANVAS_ID)
			.getContext('2d');

	/**
	 * カラーマネージャ
	 * 
	 * @type {ThreeEyeGameColorManage}
	 */
	this.color_mng = new ThreeEyeGameColorManage();

	/**
	 * ステータスマネージャ
	 * 
	 * @type {ThreeEyeGameStatusManage}
	 */
	this.status_mng = new ThreeEyeGameStatusManage();
}

/**
 * マス描画
 * 
 * @param {ThreeEyeGameMassData}
 *            data マス情報
 */
ThreeEyeGameMassDrawer.prototype.draw = function(data) {

	var x = data.getX();
	var y = data.getY();
	var color = data.getBackColor();

	// 背景色描画
	this.drawBackColor(x, y, color);
	// 枠線描画
	this.drawBorder(x, y);
	// マス状態描画
	this.drawStatus(x, y, data.getMassStatus());
};

/**
 * マス枠線描画
 * 
 * @param {number}
 *            x 左上X座標
 * @param {number}
 *            y 左上Y座標
 */
ThreeEyeGameMassDrawer.prototype.drawBorder = function(x, y) {
	this.ctx.beginPath();
	// 色設定
	this.ctx.strokeStyle = this.color_mng
			.getColor(THREE_EYE_GAME_DEFINE.COLOR_BLACK);
	// 短径描画
	this.ctx.strokeRect(x, y, THREE_EYE_GAME_DEFINE.MASS_WIDTH,
			THREE_EYE_GAME_DEFINE.MASS_HEIGHT);
	this.ctx.closePath();
};

/**
 * マス状態描画
 * 
 * @param {number}
 *            x 左上X座標
 * @param {number}
 *            y 左上Y座標
 * @param {number}
 *            status マス状態
 */
ThreeEyeGameMassDrawer.prototype.drawStatus = function(x, y, status) {

	var text = this.status_mng.getStatus(status);

	this.ctx.beginPath();
	// フォント設定
	this.ctx.font = "40px sans-serif";
	// 上下中央に描画するため、middleに設定
	this.ctx.textBaseline = "middle";
	// 色設定
	this.ctx.fillStyle = this.color_mng
			.getColor(THREE_EYE_GAME_DEFINE.COLOR_BLACK);

	// 中心座標取得
	var center_x = x + THREE_EYE_GAME_DEFINE.MASS_WIDTH / 2;
	var center_y = y + THREE_EYE_GAME_DEFINE.MASS_HEIGHT / 2;

	// 文字の中央幅取得
	var text_center_x = this.getTextWidth(this.ctx, text) / 2;

	// 描画座標を取得
	var draw_x = center_x - text_center_x;
	var draw_y = center_y;

	// マス状態描画
	this.ctx.fillText(text, draw_x, draw_y);

	this.ctx.closePath();
};

/**
 * マス背景色描画
 * 
 * @param {number}
 *            x 左上X座標
 * @param {number}
 *            y 左上Y座標
 * @param {number}
 *            color 色
 */
ThreeEyeGameMassDrawer.prototype.drawBackColor = function(x, y, color) {

	this.ctx.beginPath();
	// 色取得
	this.ctx.fillStyle = this.color_mng.getColor(color);
	// 背景色描画
	this.ctx.fillRect(x, y, THREE_EYE_GAME_DEFINE.MASS_WIDTH,
			THREE_EYE_GAME_DEFINE.MASS_HEIGHT);
	this.ctx.closePath();
};

/**
 * マス描画コンテキスト取得
 * 
 * @param {ThreeEyeGameMassData}
 *            data マス情報
 * @return {CanvasRenderingContext2D} コンテキスト
 */
ThreeEyeGameMassDrawer.prototype.getPathContext = function(data) {
	// マス左上座標取得
	var x = data.getX();
	var y = data.getY();

	this.ctx.beginPath();
	// 短径描画
	this.ctx.rect(x, y, THREE_EYE_GAME_DEFINE.MASS_WIDTH,
			THREE_EYE_GAME_DEFINE.MASS_HEIGHT);

	return this.ctx;
};

/**
 * マスクリック判定 マスがクリックされたか判定する
 * 
 * @param {Event}
 *            e マウスイベント
 * @param {ThreeEyeGameMassData}
 *            data マス情報
 * @return {boolean} クリックされたかどうか
 */
ThreeEyeGameMassDrawer.prototype.isClickMass = function(e, data) {
	// マスのパスを取得
	var ctx = this.getPathContext(data);
	
	var cross = new CROSS();
	
	var x = cross.getOffsetX(e);
	var y = cross.getOffsetY(e);
	
	// マスの中にマウスが入っているか判定
	var isClick = ctx.isPointInPath(x, y);

	return isClick;
};

/**
 * テキスト描画幅取得
 * 
 * @param {CanvasRenderingContext2D}
 *            ctx コンテキスト
 * @param {string}
 *            text 表示文字列
 * @returns {number} 幅
 */
ThreeEyeGameMassDrawer.prototype.getTextWidth = function(ctx, text) {
	// メトリクス取得
	var metrics = ctx.measureText(text);
	return metrics.width;
};

/**
 * テキスト描画高さ取得
 * 
 * @param {CanvasRenderingContext2D}
 *            ctx コンテキスト
 * @param {string}
 *            text 表示文字列
 * @returns {number} 高さ
 */
ThreeEyeGameMassDrawer.prototype.getTextHeight = function(ctx, text) {

	var spanElement = document.createElement("div");
	var bodyElement = document.getElementById("text_height");
	bodyElement.appendChild(spanElement);
	spanElement.innerText = text;
	spanElement.style.font = ctx.font;

	var emHeight = spanElement.offsetHeight;
	bodyElement
			.removeChild(bodyElement.childNodes[bodyElement.childNodes.length - 1]);
	return emHeight;
};

